<?php
/**
 * Secure Contact Form Handler with Anti-Spam Protection
 * InfiniteLoop Innovations
 *
 * Security Features:
 * - Rate limiting (session-based)
 * - Honeypot field detection
 * - Time-based submission check
 * - Input validation and sanitization
 * - CSRF token verification
 * - Email validation
 * - Suspicious content detection
 * - IP-based rate limiting
 */

session_start();

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

require 'vendor/autoload.php';
require './vendor/phpmailer/phpmailer/src/Exception.php';
require './vendor/phpmailer/phpmailer/src/PHPMailer.php';
require './vendor/phpmailer/phpmailer/src/SMTP.php';

// Configuration
define('MAX_SUBMISSIONS_PER_HOUR', 3);
define('MIN_SUBMISSION_TIME', 3); // seconds - prevents instant bot submissions
define('MAX_MESSAGE_LENGTH', 2000);
define('ENABLE_IP_BLOCKING', true);

/**
 * Anti-Spam Protection Functions
 */

// Check rate limiting
function checkRateLimit() {
    $now = time();

    // Initialize submission tracking
    if (!isset($_SESSION['submissions'])) {
        $_SESSION['submissions'] = [];
    }

    // Remove submissions older than 1 hour
    $_SESSION['submissions'] = array_filter($_SESSION['submissions'], function($timestamp) use ($now) {
        return ($now - $timestamp) < 3600;
    });

    // Check if limit exceeded
    if (count($_SESSION['submissions']) >= MAX_SUBMISSIONS_PER_HOUR) {
        return false;
    }

    return true;
}

// Record submission
function recordSubmission() {
    $_SESSION['submissions'][] = time();
}

// Verify honeypot field (should be empty)
function checkHoneypot() {
    return empty($_POST['website']) && empty($_POST['url']) && empty($_POST['company']);
}

// Check submission timing
function checkSubmissionTiming() {
    if (!isset($_SESSION['form_loaded_time'])) {
        return false;
    }

    $elapsed = time() - $_SESSION['form_loaded_time'];
    return $elapsed >= MIN_SUBMISSION_TIME;
}

// Verify CSRF token
function verifyCsrfToken() {
    if (!isset($_POST['csrf_token']) || !isset($_SESSION['csrf_token'])) {
        return false;
    }
    return hash_equals($_SESSION['csrf_token'], $_POST['csrf_token']);
}

// Validate email
function validateEmail($email) {
    // Check format
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        return false;
    }

    // Check for suspicious patterns
    $suspicious_patterns = [
        '/[0-9]{10,}/', // Too many numbers
        '/(.)\1{5,}/', // Repeated characters
        '/@(tempmail|throwaway|guerrillamail|10minutemail|mailinator)\./', // Temporary email services
    ];

    foreach ($suspicious_patterns as $pattern) {
        if (preg_match($pattern, $email)) {
            return false;
        }
    }

    return true;
}

// Check for spam content
function checkSpamContent($message) {
    $spam_keywords = [
        'viagra', 'cialis', 'casino', 'lottery', 'bitcoin',
        'crypto', 'investment opportunity', 'make money fast',
        'click here', 'buy now', 'limited time', 'act now',
        'increase sales', 'seo services', 'backlinks', 'guarantee'
    ];

    $message_lower = strtolower($message);

    foreach ($spam_keywords as $keyword) {
        if (strpos($message_lower, $keyword) !== false) {
            return false;
        }
    }

    // Check for excessive links
    if (substr_count($message, 'http') > 2) {
        return false;
    }

    return true;
}

// Sanitize input
function sanitizeInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

// IP-based blocking
function checkIpBlocked() {
    if (!ENABLE_IP_BLOCKING) {
        return false;
    }

    $ip = $_SERVER['REMOTE_ADDR'];
    $blocked_ips_file = 'blocked_ips.txt';

    if (file_exists($blocked_ips_file)) {
        $blocked_ips = file($blocked_ips_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        return in_array($ip, $blocked_ips);
    }

    return false;
}

// Log suspicious activity
function logSuspiciousActivity($reason) {
    $log_file = 'spam_log.txt';
    $ip = $_SERVER['REMOTE_ADDR'];
    $timestamp = date('Y-m-d H:i:s');
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';

    $log_entry = "[$timestamp] IP: $ip | Reason: $reason | User-Agent: $user_agent\n";
    file_put_contents($log_file, $log_entry, FILE_APPEND);
}

/**
 * Main Form Processing
 */

// Check if form was submitted
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die('Invalid request method');
}

// Security Checks
$errors = [];

// 1. Check IP blocking
if (checkIpBlocked()) {
    logSuspiciousActivity('Blocked IP attempted submission');
    die('Access denied');
}

// 2. Verify CSRF token
if (!verifyCsrfToken()) {
    $errors[] = 'Security validation failed';
    logSuspiciousActivity('CSRF token validation failed');
}

// 3. Check honeypot
if (!checkHoneypot()) {
    logSuspiciousActivity('Honeypot field filled');
    // Don't show error - just silently fail for bots
    echo '<script>';
    echo 'alert("Message has been sent.");';
    echo 'window.location.href = "https://infiniteloopinno.com/";';
    echo '</script>';
    exit;
}

// 4. Check submission timing
if (!checkSubmissionTiming()) {
    $errors[] = 'Form submitted too quickly';
    logSuspiciousActivity('Submission too fast');
}

// 5. Check rate limiting
if (!checkRateLimit()) {
    $errors[] = 'Too many submissions. Please try again later.';
    logSuspiciousActivity('Rate limit exceeded');
}

// Validate and sanitize inputs
$name = isset($_POST['name']) ? sanitizeInput($_POST['name']) : '';
$email = isset($_POST['email']) ? sanitizeInput($_POST['email']) : '';
$phone = isset($_POST['phone']) ? sanitizeInput($_POST['phone']) : '';
$subject = isset($_POST['subject']) ? sanitizeInput($_POST['subject']) : '';
$message = isset($_POST['message']) ? sanitizeInput($_POST['message']) : '';

// Validation
if (empty($name) || strlen($name) < 2) {
    $errors[] = 'Please provide a valid name';
}

if (!validateEmail($email)) {
    $errors[] = 'Please provide a valid email address';
    logSuspiciousActivity('Invalid email: ' . $email);
}

if (empty($subject) || strlen($subject) < 3) {
    $errors[] = 'Please provide a subject';
}

if (empty($message) || strlen($message) < 10) {
    $errors[] = 'Please provide a message (minimum 10 characters)';
}

if (strlen($message) > MAX_MESSAGE_LENGTH) {
    $errors[] = 'Message is too long (maximum ' . MAX_MESSAGE_LENGTH . ' characters)';
}

if (!checkSpamContent($message)) {
    $errors[] = 'Message contains inappropriate content';
    logSuspiciousActivity('Spam content detected');
}

// If there are errors, display them
if (!empty($errors)) {
    echo '<script>';
    echo 'alert("Error: ' . implode('\\n', $errors) . '");';
    echo 'window.history.back();';
    echo '</script>';
    exit;
}

// All checks passed - send email
try {
    $mail = new PHPMailer(true);

    // Server settings
    $mail->isSMTP();
    $mail->Host       = 'premium261.web-hosting.com';
    $mail->SMTPAuth   = true;
    $mail->Username   = 'hello@infiniteloopinno.com';
    $mail->Password   = 'f!%nG2,I&?LE';
    $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
    $mail->Port       = 465;

    $from = "hello@infiniteloopinno.com";
    $namefrom = "Contact Form Website";
    $mail->setFrom($from, $namefrom);

    $receiving_email_address = 'info@infiniteloopinno.com';
    $mail->addAddress($receiving_email_address, 'InfiniteLoop Innovations');

    // Reply-To for easier responses
    $mail->addReplyTo($email, $name);

    // Content
    $mail->isHTML(true);
    $mail->Subject = "Contact Form: " . $subject;

    // Enhanced email body with better formatting
    $mail->Body = "
    <html>
    <head>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 5px 5px 0 0; }
            .content { background: #f9f9f9; padding: 20px; }
            .field { margin-bottom: 15px; }
            .label { font-weight: bold; color: #667eea; }
            .value { margin-top: 5px; }
            .footer { background: #f0f0f0; padding: 10px; text-align: center; font-size: 12px; color: #666; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h2>New Contact Form Submission</h2>
            </div>
            <div class='content'>
                <div class='field'>
                    <div class='label'>Name:</div>
                    <div class='value'>" . htmlspecialchars($name) . "</div>
                </div>
                <div class='field'>
                    <div class='label'>Email:</div>
                    <div class='value'>" . htmlspecialchars($email) . "</div>
                </div>
                <div class='field'>
                    <div class='label'>Phone:</div>
                    <div class='value'>" . htmlspecialchars($phone) . "</div>
                </div>
                <div class='field'>
                    <div class='label'>Subject:</div>
                    <div class='value'>" . htmlspecialchars($subject) . "</div>
                </div>
                <div class='field'>
                    <div class='label'>Message:</div>
                    <div class='value'>" . nl2br(htmlspecialchars($message)) . "</div>
                </div>
            </div>
            <div class='footer'>
                Submitted on: " . date('Y-m-d H:i:s') . "<br>
                IP Address: " . $_SERVER['REMOTE_ADDR'] . "
            </div>
        </div>
    </body>
    </html>
    ";

    // Plain text alternative
    $mail->AltBody = "New Contact Form Submission\n\n" .
                     "Name: $name\n" .
                     "Email: $email\n" .
                     "Phone: $phone\n" .
                     "Subject: $subject\n\n" .
                     "Message:\n$message\n\n" .
                     "Submitted: " . date('Y-m-d H:i:s') . "\n" .
                     "IP: " . $_SERVER['REMOTE_ADDR'];

    // Send the email
    $mail->send();

    // Record successful submission
    recordSubmission();

    // Log successful submission (for analytics)
    $success_log = 'successful_submissions.txt';
    $log_entry = date('Y-m-d H:i:s') . " | Email: $email | IP: " . $_SERVER['REMOTE_ADDR'] . "\n";
    file_put_contents($success_log, $log_entry, FILE_APPEND);

    // Success response
    echo '<script>';
    echo 'alert("Thank you! Your message has been sent successfully. We will get back to you soon.");';
    echo 'window.location.href = "https://infiniteloopinno.com/";';
    echo '</script>';

} catch (Exception $e) {
    // Log error
    error_log("Contact form error: " . $mail->ErrorInfo);

    echo '<script>';
    echo 'alert("Sorry, there was an error sending your message. Please try again later or contact us directly at info@infiniteloopinno.com");';
    echo 'window.history.back();';
    echo '</script>';
}
?>
