/* ============================================
   Modern UI Interactive Features
   InfiniteLoop Innovations
   ============================================ */

(function() {
    'use strict';

    // ========== Utility Functions ==========

    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    function throttle(func, limit) {
        let inThrottle;
        return function() {
            const args = arguments;
            const context = this;
            if (!inThrottle) {
                func.apply(context, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    }

    // ========== Loading Screen ==========

    function initLoader() {
        window.addEventListener('load', function() {
            const loader = document.querySelector('.modern-loader');
            if (loader) {
                setTimeout(() => {
                    loader.classList.add('hidden');
                    setTimeout(() => {
                        loader.style.display = 'none';
                    }, 500);
                }, 1000);
            }
        });
    }

    // ========== Dark Mode Toggle ==========

    function initDarkMode() {
        const darkModeToggle = document.getElementById('darkModeToggle');
        const currentTheme = localStorage.getItem('theme') || 'light';

        document.documentElement.setAttribute('data-theme', currentTheme);

        if (darkModeToggle) {
            const icon = darkModeToggle.querySelector('i');
            updateDarkModeIcon(icon, currentTheme);

            darkModeToggle.addEventListener('click', function() {
                const currentTheme = document.documentElement.getAttribute('data-theme');
                const newTheme = currentTheme === 'dark' ? 'light' : 'dark';

                document.documentElement.setAttribute('data-theme', newTheme);
                localStorage.setItem('theme', newTheme);
                updateDarkModeIcon(icon, newTheme);

                // Add animation
                this.style.transform = 'rotate(360deg)';
                setTimeout(() => {
                    this.style.transform = 'rotate(0deg)';
                }, 300);
            });
        }
    }

    function updateDarkModeIcon(icon, theme) {
        if (icon) {
            icon.className = theme === 'dark' ? 'icon-sun' : 'icon-moon';
        }
    }

    // ========== Scroll Progress Bar ==========

    function initScrollProgress() {
        const progressBar = document.getElementById('scrollProgress');
        if (!progressBar) return;

        window.addEventListener('scroll', throttle(function() {
            const winScroll = document.body.scrollTop || document.documentElement.scrollTop;
            const height = document.documentElement.scrollHeight - document.documentElement.clientHeight;
            const scrolled = (winScroll / height) * 100;
            progressBar.style.width = scrolled + '%';
        }, 50));
    }

    // ========== Scroll Reveal Animations ==========

    function initScrollReveal() {
        const reveals = document.querySelectorAll('.reveal, .reveal-left, .reveal-right, .reveal-zoom');

        const revealObserver = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.classList.add('active');
                }
            });
        }, {
            threshold: 0.15,
            rootMargin: '0px 0px -50px 0px'
        });

        reveals.forEach(reveal => {
            revealObserver.observe(reveal);
        });

        // Heading underline animation
        const headings = document.querySelectorAll('.heading-section');
        const headingObserver = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.classList.add('in-view');
                }
            });
        }, { threshold: 0.5 });

        headings.forEach(heading => {
            headingObserver.observe(heading);
        });
    }

    // ========== Stats Counter Animation ==========

    function initStatsCounter() {
        const counters = document.querySelectorAll('.stat-number');

        const animateCounter = (counter) => {
            const target = parseInt(counter.getAttribute('data-target'));
            const duration = 2000;
            const increment = target / (duration / 16);
            let current = 0;

            const updateCounter = () => {
                current += increment;
                if (current < target) {
                    counter.textContent = Math.floor(current);
                    requestAnimationFrame(updateCounter);
                } else {
                    counter.textContent = target;
                }
            };

            updateCounter();
        };

        const counterObserver = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting && !entry.target.classList.contains('counted')) {
                    animateCounter(entry.target);
                    entry.target.classList.add('counted');
                }
            });
        }, { threshold: 0.5 });

        counters.forEach(counter => {
            counterObserver.observe(counter);
        });
    }

    // ========== 3D Tilt Effect ==========

    function init3DTilt() {
        const tiltElements = document.querySelectorAll('.services, .fh5co-blog, .fh5co-team');

        tiltElements.forEach(element => {
            element.addEventListener('mousemove', function(e) {
                const rect = this.getBoundingClientRect();
                const x = e.clientX - rect.left;
                const y = e.clientY - rect.top;

                const centerX = rect.width / 2;
                const centerY = rect.height / 2;

                const rotateX = (y - centerY) / 10;
                const rotateY = (centerX - x) / 10;

                this.style.transform = `perspective(1000px) rotateX(${rotateX}deg) rotateY(${rotateY}deg) scale3d(1.02, 1.02, 1.02)`;
            });

            element.addEventListener('mouseleave', function() {
                this.style.transform = 'perspective(1000px) rotateX(0) rotateY(0) scale3d(1, 1, 1)';
            });
        });
    }

    // ========== Button Ripple Effect ==========

    function initButtonRipple() {
        const buttons = document.querySelectorAll('.btn');

        buttons.forEach(button => {
            button.classList.add('btn-ripple');

            button.addEventListener('click', function(e) {
                const ripple = document.createElement('span');
                const rect = this.getBoundingClientRect();
                const size = Math.max(rect.width, rect.height);
                const x = e.clientX - rect.left - size / 2;
                const y = e.clientY - rect.top - size / 2;

                ripple.style.width = ripple.style.height = size + 'px';
                ripple.style.left = x + 'px';
                ripple.style.top = y + 'px';
                ripple.classList.add('ripple-effect');

                this.appendChild(ripple);

                setTimeout(() => {
                    ripple.remove();
                }, 600);
            });
        });
    }

    // ========== Smooth Scroll ==========

    function initSmoothScroll() {
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function(e) {
                const href = this.getAttribute('href');
                if (href !== '#' && href.length > 1) {
                    const target = document.querySelector(href);
                    if (target) {
                        e.preventDefault();
                        target.scrollIntoView({
                            behavior: 'smooth',
                            block: 'start'
                        });
                    }
                }
            });
        });
    }

    // ========== Enhanced Navigation ==========

    function initEnhancedNav() {
        const header = document.getElementById('fh5co-header-section');
        let lastScroll = 0;

        window.addEventListener('scroll', throttle(function() {
            const currentScroll = window.pageYOffset;

            if (currentScroll > 100) {
                header.classList.add('scrolled');
            } else {
                header.classList.remove('scrolled');
            }

            // Hide/show on scroll
            if (currentScroll > lastScroll && currentScroll > 500) {
                header.style.transform = 'translateY(-100%)';
            } else {
                header.style.transform = 'translateY(0)';
            }

            lastScroll = currentScroll;
        }, 100));
    }

    // ========== Particle Background ==========

    function initParticles() {
        if (typeof particlesJS !== 'undefined') {
            const particleContainers = document.querySelectorAll('#particles-js');
            particleContainers.forEach(container => {
                particlesJS(container.id, {
                    particles: {
                        number: {
                            value: 80,
                            density: {
                                enable: true,
                                value_area: 800
                            }
                        },
                        color: {
                            value: '#ffffff'
                        },
                        shape: {
                            type: 'circle',
                        },
                        opacity: {
                            value: 0.5,
                            random: false,
                        },
                        size: {
                            value: 3,
                            random: true,
                        },
                        line_linked: {
                            enable: true,
                            distance: 150,
                            color: '#ffffff',
                            opacity: 0.4,
                            width: 1
                        },
                        move: {
                            enable: true,
                            speed: 2,
                            direction: 'none',
                            random: false,
                            straight: false,
                            out_mode: 'out',
                            bounce: false,
                        }
                    },
                    interactivity: {
                        detect_on: 'canvas',
                        events: {
                            onhover: {
                                enable: true,
                                mode: 'grab'
                            },
                            onclick: {
                                enable: true,
                                mode: 'push'
                            },
                            resize: true
                        },
                        modes: {
                            grab: {
                                distance: 140,
                                line_linked: {
                                    opacity: 1
                                }
                            },
                            push: {
                                particles_nb: 4
                            }
                        }
                    },
                    retina_detect: true
                });
            });
        }
    }

    // ========== GSAP Animations ==========

    function initGSAPAnimations() {
        if (typeof gsap !== 'undefined') {
            // Animate hero text
            gsap.from('.fh5co-cover .desc h2', {
                duration: 1,
                y: 100,
                opacity: 0,
                ease: 'power4.out',
                delay: 0.5
            });

            gsap.from('.fh5co-cover .desc span', {
                duration: 1,
                y: 50,
                opacity: 0,
                ease: 'power4.out',
                delay: 0.8
            });

            // Stagger animations for cards
            gsap.from('.services', {
                scrollTrigger: {
                    trigger: '.services',
                    start: 'top 80%',
                },
                duration: 0.8,
                y: 100,
                opacity: 0,
                stagger: 0.2,
                ease: 'power3.out'
            });

            // Team members animation
            gsap.from('.fh5co-team', {
                scrollTrigger: {
                    trigger: '.fh5co-team',
                    start: 'top 80%',
                },
                duration: 1,
                scale: 0.8,
                opacity: 0,
                stagger: 0.3,
                ease: 'back.out(1.7)'
            });
        }
    }

    // ========== Gradient Text Animation ==========

    function initGradientTextAnimation() {
        const gradientTexts = document.querySelectorAll('.gradient-text');
        gradientTexts.forEach(text => {
            text.style.backgroundSize = '200% 200%';
        });
    }

    // ========== Image Lazy Loading ==========

    function initLazyLoading() {
        const images = document.querySelectorAll('img[data-src]');

        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    img.src = img.dataset.src;
                    img.classList.add('loaded');
                    observer.unobserve(img);
                }
            });
        });

        images.forEach(img => imageObserver.observe(img));
    }

    // ========== Custom Cursor ==========

    function initCustomCursor() {
        const cursorDot = document.querySelector('.cursor-dot');
        const cursorOutline = document.querySelector('.cursor-outline');

        if (!cursorDot || !cursorOutline) return;

        let mouseX = 0, mouseY = 0;
        let outlineX = 0, outlineY = 0;

        document.addEventListener('mousemove', (e) => {
            mouseX = e.clientX;
            mouseY = e.clientY;

            cursorDot.style.left = mouseX + 'px';
            cursorDot.style.top = mouseY + 'px';
        });

        // Smooth follow for outline
        function animateOutline() {
            outlineX += (mouseX - outlineX) * 0.15;
            outlineY += (mouseY - outlineY) * 0.15;

            cursorOutline.style.left = outlineX + 'px';
            cursorOutline.style.top = outlineY + 'px';

            requestAnimationFrame(animateOutline);
        }
        animateOutline();

        // Expand cursor on interactive elements
        document.querySelectorAll('a, button, .btn').forEach(el => {
            el.addEventListener('mouseenter', () => {
                cursorOutline.style.transform = 'scale(1.5)';
            });
            el.addEventListener('mouseleave', () => {
                cursorOutline.style.transform = 'scale(1)';
            });
        });
    }

    // ========== Parallax Effect ==========

    function initParallax() {
        const parallaxElements = document.querySelectorAll('[data-parallax]');

        window.addEventListener('scroll', throttle(function() {
            const scrolled = window.pageYOffset;

            parallaxElements.forEach(element => {
                const speed = element.dataset.parallax || 0.5;
                const yPos = -(scrolled * speed);
                element.style.transform = `translateY(${yPos}px)`;
            });
        }, 16));
    }

    // ========== Form Enhancements ==========

    function initFormEnhancements() {
        const formInputs = document.querySelectorAll('.form-control');

        formInputs.forEach(input => {
            // Floating label effect
            input.addEventListener('focus', function() {
                this.parentElement.classList.add('focused');
            });

            input.addEventListener('blur', function() {
                if (this.value === '') {
                    this.parentElement.classList.remove('focused');
                }
            });

            // Auto-resize textarea
            if (input.tagName === 'TEXTAREA') {
                input.addEventListener('input', function() {
                    this.style.height = 'auto';
                    this.style.height = this.scrollHeight + 'px';
                });
            }
        });
    }

    // ========== Toast Notifications ==========

    function showToast(message, type = 'info', duration = 3000) {
        const toast = document.createElement('div');
        toast.className = `toast toast-${type}`;
        toast.textContent = message;
        toast.style.cssText = `
            position: fixed;
            bottom: 30px;
            right: 30px;
            padding: 15px 25px;
            background: var(--primary-gradient);
            color: white;
            border-radius: var(--radius-md);
            box-shadow: var(--shadow-xl);
            z-index: 10000;
            animation: slideInRight 0.3s ease-out;
        `;

        document.body.appendChild(toast);

        setTimeout(() => {
            toast.style.animation = 'slideOutRight 0.3s ease-out';
            setTimeout(() => toast.remove(), 300);
        }, duration);
    }

    // ========== Mobile Menu Enhancement ==========

    function initMobileMenuEnhancement() {
        const menuToggle = document.querySelector('.fh5co-nav-toggle');
        const offcanvasMenu = document.getElementById('offcanvas-menu');

        if (menuToggle && offcanvasMenu) {
            menuToggle.addEventListener('click', function() {
                // Add smooth transition
                offcanvasMenu.style.transition = 'transform 0.3s ease-out';
            });
        }
    }

    // ========== Performance Monitoring ==========

    function logPerformance() {
        if (window.performance) {
            const perfData = window.performance.timing;
            const pageLoadTime = perfData.loadEventEnd - perfData.navigationStart;
            console.log('Page load time:', pageLoadTime + 'ms');
        }
    }

    // ========== Initialize All Features ==========

    function init() {
        // Core features
        initLoader();
        initDarkMode();
        initScrollProgress();
        initSmoothScroll();
        initEnhancedNav();

        // Visual enhancements
        initScrollReveal();
        initGradientTextAnimation();
        init3DTilt();
        initButtonRipple();

        // Animations
        initStatsCounter();
        initParallax();

        // External library features
        initParticles();
        initGSAPAnimations();

        // Form and UI
        initFormEnhancements();
        initMobileMenuEnhancement();

        // Advanced features
        initLazyLoading();
        initCustomCursor();

        // Performance
        logPerformance();

        // Expose showToast globally
        window.showToast = showToast;
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

    // Add resize handler
    window.addEventListener('resize', debounce(function() {
        // Reinit responsive features if needed
    }, 250));

})();
