/**
 * Contact Form Security and Validation
 * Anti-Spam JavaScript Protection
 */

(function() {
    'use strict';

    // Generate CSRF token on page load
    function generateCsrfToken() {
        const array = new Uint8Array(32);
        window.crypto.getRandomValues(array);
        return Array.from(array, byte => byte.toString(16).padStart(2, '0')).join('');
    }

    // Store form load time
    const formLoadTime = Date.now();

    // Initialize form security
    function initFormSecurity() {
        const csrfTokenInput = document.getElementById('csrf_token');
        const contactForm = document.getElementById('contactForm');

        if (csrfTokenInput) {
            // Generate and set CSRF token
            const token = generateCsrfToken();
            csrfTokenInput.value = token;

            // Store in sessionStorage for verification
            sessionStorage.setItem('csrf_token', token);
        }

        if (contactForm) {
            // Character counter for message field
            const messageField = document.getElementById('message');
            const charCount = document.getElementById('charCount');

            if (messageField && charCount) {
                messageField.addEventListener('input', function() {
                    const length = this.value.length;
                    charCount.textContent = `${length} / 2000 characters`;

                    if (length > 2000) {
                        charCount.style.color = '#dc3545';
                    } else if (length > 1800) {
                        charCount.style.color = '#ffc107';
                    } else {
                        charCount.style.color = '#6c757d';
                    }
                });
            }

            // Disable submit button temporarily to prevent double submission
            contactForm.addEventListener('submit', function() {
                const submitBtn = document.getElementById('submitBtn');
                if (submitBtn) {
                    submitBtn.disabled = true;
                    submitBtn.value = 'Sending...';

                    // Re-enable after 3 seconds in case of error
                    setTimeout(() => {
                        submitBtn.disabled = false;
                        submitBtn.value = 'Send Message';
                    }, 3000);
                }
            });
        }
    }

    // Form validation function
    window.validateForm = function() {
        const form = document.getElementById('contactForm');

        // Check honeypot fields (should be empty)
        const honeypotWebsite = document.querySelector('input[name="website"]');
        const honeypotUrl = document.querySelector('input[name="url"]');
        const honeypotCompany = document.querySelector('input[name="company"]');

        if ((honeypotWebsite && honeypotWebsite.value) ||
            (honeypotUrl && honeypotUrl.value) ||
            (honeypotCompany && honeypotCompany.value)) {
            // Silent fail for bots
            return false;
        }

        // Check submission timing (must be at least 3 seconds after load)
        const timeSinceLoad = (Date.now() - formLoadTime) / 1000;
        if (timeSinceLoad < 3) {
            alert('Please take a moment to review your message before submitting.');
            return false;
        }

        // Validate inputs
        const name = document.getElementById('name').value.trim();
        const email = document.getElementById('email').value.trim();
        const phone = document.getElementById('phone').value.trim();
        const subject = document.getElementById('subject').value.trim();
        const message = document.getElementById('message').value.trim();

        // Name validation
        if (name.length < 2) {
            alert('Please enter your full name (at least 2 characters).');
            return false;
        }

        // Email validation
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(email)) {
            alert('Please enter a valid email address.');
            return false;
        }

        // Check for suspicious email patterns
        const suspiciousEmailDomains = ['tempmail', 'throwaway', 'guerrillamail', '10minutemail', 'mailinator'];
        const emailDomain = email.split('@')[1]?.toLowerCase() || '';

        for (const domain of suspiciousEmailDomains) {
            if (emailDomain.includes(domain)) {
                alert('Please use a permanent email address.');
                return false;
            }
        }

        // Phone validation
        if (phone.length < 7) {
            alert('Please enter a valid phone number.');
            return false;
        }

        // Subject validation
        if (subject.length < 3) {
            alert('Please provide a more descriptive subject.');
            return false;
        }

        // Message validation
        if (message.length < 10) {
            alert('Please provide a more detailed message (at least 10 characters).');
            return false;
        }

        if (message.length > 2000) {
            alert('Message is too long (maximum 2000 characters).');
            return false;
        }

        // Check for spam keywords
        const spamKeywords = [
            'viagra', 'cialis', 'casino', 'lottery', 'bitcoin',
            'crypto', 'click here', 'buy now', 'act now', 'guarantee',
            'make money fast', 'increase sales', 'seo services'
        ];

        const messageLower = message.toLowerCase();
        for (const keyword of spamKeywords) {
            if (messageLower.includes(keyword)) {
                alert('Your message contains inappropriate content. Please revise and try again.');
                return false;
            }
        }

        // Check for excessive links
        const urlCount = (message.match(/http/gi) || []).length;
        if (urlCount > 2) {
            alert('Please limit the number of links in your message to 2.');
            return false;
        }

        // Check for repeated characters (potential spam)
        if (/(.)\1{10,}/.test(message)) {
            alert('Your message contains unusual patterns. Please revise and try again.');
            return false;
        }

        // All validations passed
        return true;
    };

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initFormSecurity);
    } else {
        initFormSecurity();
    }

    // Add real-time email validation styling
    const emailInput = document.getElementById('email');
    if (emailInput) {
        emailInput.addEventListener('blur', function() {
            const email = this.value.trim();
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

            if (email && !emailRegex.test(email)) {
                this.style.borderColor = '#dc3545';
            } else {
                this.style.borderColor = '';
            }
        });
    }

    // Add real-time phone validation styling
    const phoneInput = document.getElementById('phone');
    if (phoneInput) {
        phoneInput.addEventListener('input', function() {
            // Allow only numbers, spaces, +, -, (, )
            this.value = this.value.replace(/[^0-9\s\+\-\(\)]/g, '');
        });
    }

    // Prevent rapid form resubmission
    let lastSubmitTime = 0;
    const MIN_SUBMIT_INTERVAL = 10000; // 10 seconds between submissions

    if (document.getElementById('contactForm')) {
        document.getElementById('contactForm').addEventListener('submit', function(e) {
            const now = Date.now();
            if (now - lastSubmitTime < MIN_SUBMIT_INTERVAL) {
                e.preventDefault();
                alert('Please wait a moment before submitting again.');
                return false;
            }
            lastSubmitTime = now;
        });
    }

})();
